<?php
/* --------------------------------------------------------------
   GambioHubSavedPaymentController.inc.php 2025-07-29
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2025 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/
use \HubPublic\Http\CurlRequest;

class HubVaultController extends HttpViewController
{
    /**
     * @var LanguageTextManager
     */
    protected $languageTextManager;
    public $customerId = null;
    protected $hubAuthenticated = false;
    public $logger = null;
    public $clientKey  = null;

    public function init()
    {
        $this->languageTextManager = MainFactory::create(
            'LanguageTextManager',
            'gambio_hub',
            $_SESSION['languages_id']
        );
    }

    public function proceed(HttpContextInterface $httpContext)
    {
        if (isset($_SESSION['customer_id']) === false) {
            xtc_redirect(
                xtc_href_link(
                    FILENAME_LOGIN,
                    'return_url=' . rawurlencode(GM_HTTP_SERVER . gm_get_env_info('REQUEST_URI')),
                    'SSL'
                )
            );
        }


        $serverData             = $this->httpContextReader->getServerData($httpContext);
        $clientKey              = (string)gm_get_conf('GAMBIO_HUB_CLIENT_KEY');
        $this->hubAuthenticated = !empty($clientKey)
                                  && !empty($serverData['HTTP_X_CLIENT_KEY'])
                                  && $clientKey === $serverData['HTTP_X_CLIENT_KEY'];
        $this->logger           = LogControl::get_instance();
        $this->customerId       = $_SESSION['customer_id'] ?? null;
        $this->clientKey        = $clientKey;
        parent::proceed($httpContext);

    }

    /**
     * Stores the client key to the shop configuration.
     */
    public function actionDefault()
    {
        $contentControl = $this->getLayoutContentControl();

        // load view
        $contentView = MainFactory::create('HubListVaultThemeContentView');
        $contentView->set_vault_data_array($this->loadVaultData());

        // process view
        $contentControl->set_('main_content', $contentView->get_html());
        $contentControl->proceed();

        $redirectUrl = $contentControl->get_redirect_url();
        if (empty($redirectUrl) === false) {
            xtc_redirect($redirectUrl);
        }

        return MainFactory::create('HttpControllerResponse', $contentControl->get_response());
    }

    /**
     * Handle Delete vault
     * 
     * @return mixed
     */
    public function actionDelete()
    {
        $vaultId = $this->_getQueryParameter('vault_id');
        if (empty($vaultId)) {
            return $this->redirectToMainPage();
        }
        $confirmation = $this->_getQueryParameter('confirmation');
        $type = $this->_getQueryParameter('type');
        if (!empty($type) && !empty($confirmation) && $confirmation === 'true') {
            $this->doDelete($vaultId, $type);
            return $this->redirectToMainPage();
        }
        $vaultData = $this->filterByTokenId($vaultId);
        if (empty($vaultData)) {
            return $this->redirectToMainPage();
        }
        $contentView = MainFactory::create('HubDeleteVaultThemeContentView');
        $contentView->setVaultData($vaultData);
        $contentControl = $this->getLayoutContentControl();
        // process view
        $contentControl->set_('main_content', $contentView->get_html());
        $contentControl->proceed();
        $redirectUrl = $contentControl->get_redirect_url();
        if (empty($redirectUrl) === false) {
            xtc_redirect($redirectUrl);
        }

        return MainFactory::create('HttpControllerResponse', $contentControl->get_response());

    }

    /**
     * @return mixed
     */
    public function redirectToMainPage()
    {
        return MainFactory::create(
            'RedirectHttpControllerResponse',
            xtc_href_link('shop.php', 'do=HubVault', 'SSL')
        );
    }
    
    /**
     * @return LayoutContentControl
     */
    protected function getLayoutContentControl()
    {
        $GLOBALS['breadcrumb']->add(NAVBAR_TITLE_ACCOUNT, xtc_href_link(FILENAME_ACCOUNT, '', 'SSL'));
        $GLOBALS['breadcrumb']->add(
            $this->languageTextManager->get_text(
                'MODULE_PAYMENT_GAMBIO_HUB_VAULT_MANAGEMENT',
                'gambio_hub'
            ),
            xtc_href_link('shop.php', 'do=HubVault', 'SSL')
        );
        $contentControl = MainFactory::create_object('LayoutContentControl');
        $contentControl->set_data('GET', $this->_getQueryParametersCollection()->getArray());
        $contentControl->set_data('POST', $this->_getPostDataCollection()->getArray());
        $contentControl->set_('coo_breadcrumb', $GLOBALS['breadcrumb']);
        $contentControl->set_('coo_product', $GLOBALS['product']);
        $contentControl->set_('coo_xtc_price', $GLOBALS['xtPrice']);
        $contentControl->set_('c_path', $GLOBALS['cPath']);
        $contentControl->set_('request_type', $GLOBALS['request_type']);

        return $contentControl;
    }
    protected function loadVaultData()
    {
        try {
            // Get client key from configuration
            $clientKey = gm_get_conf('GAMBIO_HUB_CLIENT_KEY');
            if (empty($clientKey)) {
                return [];
            }

            // Create HubCallbackApiClient to make the API call
            $hubSettings          = MainFactory::create('HubSettings', gm_get_conf('GAMBIO_HUB_CURL_TIMEOUT'));
            $hubCallbackApiClient = MainFactory::create(
                'HubCallbackApiClient',
                MODULE_PAYMENT_GAMBIO_HUB_URL,
                new CurlRequest(),
                LogControl::get_instance(),
                $hubSettings
            );

            $query = [
                'client_key' => $clientKey
            ];

            // Prepare headers with user session data
            $headers = [];
            if (!empty($_SESSION['customer_id'])) {
                $headers[] = 'X-GX-User-ID: ' . $_SESSION['customer_id'];
            }

            // Make the API call to HubAPI
            $response = $hubCallbackApiClient->execute(
                'PayPal2Hub',
                true,
                ['source' => 'saved_payments', 'language' => $_SESSION['language_code']],
                $query,
                $headers
            );

            if ($response->getStatusCode() === 200) {
                $responseData = json_decode($response->getBody(), true);
                if (json_last_error() === JSON_ERROR_NONE &&
                    (isset($responseData['saved_cards']) || isset($responseData['saved_paypal']))
                ) {
                    return $responseData;
                }
            }
        } catch (Exception $e) {
            // Log error and set empty array
            $logControl = LogControl::get_instance();
            $logControl->notice(
                'Failed to get vault data from HubAPI: ' . $e->getMessage(),
                '',
                'hub',
                'notice',
                'USER NOTICE'
            );
        }
        return [];
    }

    /**
     * Filer vault by tokenID 
     * 
     * @param string $tokenId
     * @return array|null
     */
    private function filterByTokenId(string $tokenId): ?array
    {
        $data = $this->loadVaultData();
        // Check saved_cards first
        if (isset($data['saved_cards']) && array_key_exists($tokenId, $data['saved_cards'])) {
            return $data['saved_cards'][$tokenId];
        }

        // Check saved_paypal if not found in saved_cards
        if (isset($data['saved_paypal']) && array_key_exists($tokenId, $data['saved_paypal'])) {
            return $data['saved_paypal'][$tokenId];
        }
        // Return null if the token ID is not found in either section
        return null;
    }

    /**
     * Exec delete request to hub
     * 
     * @param string $vaultId
     * @param string $type
     * @return void
     */
    private function doDelete(string $vaultId, string $type) : void
    {
        try {
            if (empty($this->clientKey || empty($this->customerId))) {
                return;
            }
            $hubSettings = MainFactory::create('HubSettings', gm_get_conf('GAMBIO_HUB_CURL_TIMEOUT'));
            $hubCallbackApiClient = MainFactory::create(
                'HubCallbackApiClient',
                MODULE_PAYMENT_GAMBIO_HUB_URL,
                new CurlRequest(),
                $this->logger,
                $hubSettings
            );

            $query = [
                'client_key' => $this->clientKey,
                'vault_id' => $vaultId,
                'customer_id' => $this->customerId,
                'type' => $type,
            ];

            $hubCallbackApiClient->execute(
                'PayPal2Hub',
                true,
                [
                    'source' => 'delete_user_vault',
                ],
                $query
            );
            if ($type === 'paypal') {
                unset($_SESSION['customer_vault']);
            }
        } catch (Exception $exception) {
            $this->logger->notice(
                "Failed to execute delete vault Error: " . $exception->getMessage(),
                '',
                'hub',
                'notice',
                'USER NOTICE'
            );
            return;
        }
    }
}
